<?php

class BigCommerce
{
    private $client;
    private $storeId;

    public function __construct($client, $storeId) {
        $this->client = $client;
        $this->storeId = $storeId;
    }

    public static function create($client, $storeId)
    {
        $object = get_called_class();
        return new $object($client, $storeId);
    }

    function getFinanceDetails($orderId)
    {
        $response = $this->client->get('https://api.bigcommerce.com/stores/' . $this->storeId . '/v2/gift_certificates?order_id=' . $orderId);
        if ($response->getStatusCode() != 200) {
            throw new \Exception('ABORT: Something went wrong: ' . $response->getStatusCode());
        }
        $body = json_decode($response->getBody()->getContents());
        if (!isset($body[0])) {
            throw new \Exception('ABORT: Unable to find finance details');
        }
        return $body[0];
    }

    function getOrder($orderId)
    {
        $response = $this->client->get('https://api.bigcommerce.com/stores/' . $this->storeId . '/v2/orders/' . $orderId);
        if ($response->getStatusCode() != 200) {
            throw new \Exception('ABORT: Something went wrong: ' . $response->getStatusCode());
        }
        $body = json_decode($response->getBody()->getContents());
        return $body;
    }

    function getCart($cartId)
    {
        $response = $this->client->get('https://api.bigcommerce.com/stores/' . $this->storeId . '/v3/carts/' . $cartId);
        if ($response->getStatusCode() != 200) {
            throw new \Exception('ABORT: Something went wrong: ' . $response->getStatusCode());
        }
        $body = json_decode($response->getBody()->getContents());
        return $body;
    }

    function getCartData($hashSecret, $existingCart, $financeData, $loanAmount) {
        $simplePhysicalProducts = $existingCart->data->line_items->physical_items;
        $lineItems = [];
        foreach ($simplePhysicalProducts as $p) {
            $lineItems[] = [
                'product_id' => $p->product_id,
                'quantity' => $p->quantity,
            ];
        }
        $data = [
            // 'line_items' => $lineItems, // Ideally we would like to validate the line items have not changed, but the gift message is a limited size field
            'f' => $loanAmount, //finance_amount
            'df' => $financeData['deferralterm'],
            't' => $financeData['total'],
            'a' => $financeData['apr'],
            'i' => $financeData['instalment'],
            'd' => $financeData['deposit'],
            't' => $financeData['term'],
            's' => $financeData['service'],
            'r' => rand(0,100000), // A random number to keep the hash unique
        ];
        $json = [
            'line_items' => $lineItems,
            'gift_certificates' => [[
                'name' => 'Finance',
                'theme' => 'General',
                'amount' => $loanAmount,
                'quantity' => 1,
                'sender' => [
                    'name' => 'Finance (s)',
                ],
                'recipient' => [
                    'name' => 'Finance (r)',
                ],
                // Message needs to be less than 250 characters
                'message' => json_encode([
                    'hash' => hash_hmac('sha256', json_encode($data), $hashSecret),
                    'fd' => $data,
                ]),
            ]],
        ];
        return $json;
    }

    function createDraftOrder($cartData)
    {
        $response = $this->client->post('https://api.bigcommerce.com/stores/' . $this->storeId . '/v3/carts?include=redirect_urls', [
            'json' => $cartData,
        ]);
        if ($response->getStatusCode() != 201) {
            throw new \Exception('ABORT: Something went wrong: ' . $response->getStatusCode());
        }
        $body = json_decode($response->getBody()->getContents());
        return $body;
    }    
}

